 ; Ŀ
 ;   Life - an implementation of Conway's Cellular Automata game.          
 ;   Copyright 1998 by Rocket Software                                     
 ;   Contains 5 programs:                                                  
 ;   Setup: makes a grid of the desired size with cells, these can be      
 ;          erased to leave the desired arrangement.                       
 ;   Tog:   Pick a cell, toggles it on or off.                             
 ;   Life:  Runs the life program.                                         
 ;   Lsave: Saves the current life grid to a file.                         
 ;   Lread: Reads a life grid from a file.                                 
 ;                                                                         
 ; 

 ; Ŀ
 ;   Rules of Life:                                                        
 ;   If a full cell has 0, 1, or more than 4 neighbors it dies, of either  
 ;   loneliness or overcrowding respectively.                              
 ;   If it has 2 or 3 neighbors it is unchanged.                           
 ;   If an empty cell has 3 neighbors it generates a cell.                 
 ;   All cells are assumed to appear or die simultaneously.                
 ; 

 ; Ŀ
 ;   LRead - load a life data set from a file.                             
 ; 
 (DEFUN C:LREAD (/ filnam fn num str keep enam)
  (setvar "cmdecho" 0)
  (if (and (/= 1 (setq filnam (getfiled "Data File" "" "lif" 0)))
           (setq fn (open filnam "r")))
      (progn
           (setq num 0)
           (while (setq str (read-line fn))
                  (while (= (substr str 1 1) " ")
                         (setq str (substr str 2)))
                  (cond ((= (strcase (substr str 1 10) t) "grid size:")
                         (setq gridsz (read (substr str 11))))
                         ((and (/= (substr str 1 1) ";")
                               (/= str ""))
                          (setq keep (append keep (list str)))))
                  (setq num (1+ num)))
           (close fn)
           (setq pa (getpoint "Upper Left: "))
           (setq celen (getdist pa "Cell length: "))
           (setq malist (dotto pa celen gridsz))
           (command ".redraw")
           (setq pa (polar (polar pa pi (/ celen 2)) (/ pi 2) (/ celen 2)))
           (bomax pa gridsz celen)
           (setq num 0)
           (while (setq celnam (car (setq sub (nth num malist))))
                  (setq num (1+ num))
                  (if (not (member celnam keep))
                      (entdel (cadr sub)))))
      (write-line "Can't open file."))
 (princ))
 ; Ŀ
 ;   LRead end.                                                            
 ; 

 ; Ŀ
 ;   Lsave - Save a life pattern to a file.                                
 ; 
 (DEFUN C:LSAVE (/ filout str num)
  (if (and (/= 1 (setq filout (getfiled "Life Data File" "" "lif" 1)))
           (bottle filout (list
                        "Editable data file for Rocket Software's Life.lsp."
                        "Rows are A,B,C and so on starting at the top."
                        "Columns are 0,1,2 etc. from the left."
                        ""
                        (slog)))
           (setq fn (open filout "a")))
      (progn
           (setq num 0)
           (write-line "" fn)
           (setq gridsz (itoa (fix (sqrt (length malist)))))
           (write-line (strcat "Grid size: " gridsz) fn)
           (write-line "" fn)
           (while (setq str (nth num malist))
                  (if (entget (cadr str))
                      (write-line (car str) fn))
                  (setq num (1+ num)))
           (close fn))
      (write-line "Can't open file."))
 (princ))
 ; Ŀ
 ;   Lsave end.                                                            
 ; 

 ; Ŀ
 ;   Bottle - write a boxed file header.                                   
 ;   Takes no prisoners, returns nothing.                                  
 ;   Correction - takes one argument, a filename.                          
 ;   Further correction - takes another argument, list of strings to       
 ;   write, each on its own line.                                          
 ; 
 (DEFUN BOTTLE (lognam strlst / aa bb cc thestr newlst lognam fn)
  (setq aa "")
  (setq bb (strcat " ; " aa aa ""))
  (setq cc (strcat " ; " aa aa ""))
  (while (setq thestr (car strlst))
         (setq strlst (cdr strlst))
         (setq thestr (strcat " ;   " thestr))
         (while (< (strlen thestr) 76) (setq thestr (strcat thestr " ")))
         (setq thestr (strcat thestr ""))
         (setq newlst (append newlst (list thestr))))
  (setq fn (open lognam "w"))
  (princ bb fn)
  (while (setq thestr (car newlst))
         (setq newlst (cdr newlst))
         (princ (strcat "\n" thestr) fn))
  (princ (strcat "\n" cc "\n") fn)
  (close fn)
 (princ))
 ; Ŀ
 ;   Bottle end.                                                           
 ; 

 ; Ŀ
 ;   Slog - returns a text string chosen at random from a list.            
 ;   Takes no arguments, returns a string.                                 
 ; 
 (DEFUN SLOG (/ s nnum mlst)
  (setq s (* (getvar "cdate") 10000000.0))
  (setq mlst (list
  "Life, contrary to popular belief, comes before art."
  "Can you resist editing a file that says \"Do Not Edit\"?"
  "Increasing complexity does not equate to increased utility."
  "Dogs - do they all secretly revere Karl Marx?"
  "Central Planning - the first and worst mistake made by humanity."
  "An ostrich is more flambouyant than a ballpoint, if less convenient."
  "The Rhinocerous: the only naturally occurring organic paperweight."
  "CDs will soon be replaced by music recorded on pork chops."
  "If cubicles are so great, why don't managers have them at home?"
  "Why are the slats in venetian blinds always straight?"))
  (setq nnum (fix (* 10 (- s ( fix s)))))
 (nth nnum mlst))
 ; Ŀ
 ;   Slog end.                                                             
 ; 

 ; Ŀ
 ;   Tog - toggle a cell on or off.                                        
 ; 
 (DEFUN C:TOG (/ scrbxs str togpt togx togy xdist xindx ydist yindx letter
                                                              celnam cenam)
  (setq scrbxs (1- (getvar "screenboxes")))
  (setq str "Cell: ")
  (if (and pa celen malist)
      (while (setq togpt (getpoint str))
             (setq str "")
             (setq togx (car togpt))
             (setq togy (cadr togpt))
             (setq xdist (- togx (car pa)))
             (setq xindx (fix (/ xdist celen)))
             (setq ydist (- (cadr pa) togy))
             (setq yindx (fix (/ ydist celen)))
             (setq letter (chr (+ yindx 65)))
             (setq celnam (strcat letter (itoa xindx)))
             (if (setq cenam (cadr (assoc celnam malist)))
                 (progn
                      (grtext scrbxs celnam)
                      (entdel cenam))
                 (prompt (strcat "\nNo cell " celnam " found.\n"))))
      (prompt "\nNo active Life - run Setup."))
 (princ))
 ; Ŀ
 ;   Tog end.                                                              
 ; 

 ; Ŀ
 ;   Bomax - Draw a box divided into a grid.                               
 ;   Takes three arguments - an upper left corner, a grid size, and the    
 ;   side length.  Assumes the box is a square.                            
 ; 
 (DEFUN BOMAX (pa divs celen / pasav pa2 pa3 pa4)
  (setq pasav pa)
 ; Ŀ
 ;   Draw the box.                                                         
 ; 
  (setq ovrlen (* divs celen))
  (setq pa2 (polar pa 0 ovrlen))
  (setq pa3 (polar pa2 (* pi 1.5) ovrlen))
  (setq pa4 (polar pa3 pi ovrlen))
  (command ".pline" pa pa2 pa3 pa4 "c")
 ; Ŀ
 ;   And the grid lines.                                                   
 ; 
  (repeat (1- divs)
          (setq pa (polar pa (* pi 1.5) celen))
          (setq pa2 (polar pa2 (* pi 1.5) celen))
          (command ".line" pa pa2 ""))
  (setq pa pasav)
  (repeat (1- divs)
          (setq pa (polar pa 0 celen))
          (setq pa4 (polar pa4 0 celen))
          (command ".line" pa pa4 ""))
 (princ))
 ; Ŀ
 ;   Bomax end.                                                            
 ; 

 ; Ŀ
 ;   Neigh - get the names of all 8 neighbors to a names cell.             
 ;   Takes two arguments, a cell name and the grid size.                   
 ;   Returns the number of living cells in the neighbor cells.             
 ; 
 (DEFUN NEIGH (celnam gridsz / maxchr celchr celnum rowup rowdn numlft numrt
                                                        loclst num alive sub)
  (setq maxchr (chr (+ 64 gridsz)))
  (setq celchr (substr celnam 1 1))
  (setq celnum (read (substr celnam 2)))
  (setq rowup (if (= celchr "A") maxchr (chr (1- (ascii celchr)))))
  (setq rowdn (if (= celchr maxchr) "A" (chr (1+ (ascii celchr)))))
  (setq numlft (if (= celnum 0) (itoa (1- gridsz)) (itoa (1- celnum))))
  (setq numrt (if (= celnum (1- gridsz)) "0" (itoa (1+ celnum))))
  (setq loclst (list (strcat rowup numlft)         ; upper left
                     (strcat rowup numrt)          ; uright
                     (strcat rowdn numlft)         ; lower left
                     (strcat rowdn numrt)          ; lower right
                     (strcat rowup (itoa celnum))  ; top centre
                     (strcat rowdn (itoa celnum))  ; bottom centre
                     (strcat celchr numlft)        ; left
                     (strcat celchr numrt)))       ; right
  (setq num 0)
  (setq alive 0)
  (while (setq sub (nth num loclst))
         (setq num (1+ num))
         (if (entget (cadr (assoc sub malist)))
             (setq alive (1+ alive))))
 alive)
 ; Ŀ
 ;   Neigh end.                                                            
 ; 

 ; Ŀ
 ;   Dotto - draws 400 dots and saves the locations and enames to Malist.  
 ;   Takes three arguments: a start (upper left) point, the cell centre    
 ;   to centre distance, and the grid size in cells.                       
 ;   Returns a list of lists: ((cellname dot ename)...)                    
 ;   The rows are numbered A to T and the columns 0 to 19, so cell names   
 ;   are A1, A2, etc., next row B1, B2, etc.                               
 ; 
 (DEFUN DOTTO (pa celen gridsz / blip pa1 num rowchr enam colnum malist cells)
  (setq blip (getvar "blipmode"))
  (setvar "blipmode" 0)
  (setq pa1 pa)
  (setq num 1)
  (setq rowchr "A")
  (setq cells (* gridsz gridsz))
  (while (<= num cells)
         (command ".doughnut" 0 (/ celen 2) pa "")
         (setq enam (entlast))
         (if (= (setq colnum (rem num gridsz)) 0.0)
             (progn
                  (if (= colnum 0)
                      (setq colnum (1- gridsz))
                      (setq colnum (1- colnum)))
                  (setq malist (append malist
                            (list (list (strcat rowchr (itoa colnum)) enam))))
                  (setq rowchr (chr (1+ (ascii rowchr))))
                  (setq pa1 (polar pa1 (* pi 1.5) celen))
                  (setq pa pa1))
             (progn
                  (if (= colnum 0)
                      (setq colnum (1- gridsz))
                      (setq colnum (1- colnum)))
                  (setq malist (append malist
                            (list (list (strcat rowchr (itoa colnum)) enam))))
                  (setq pa (polar pa 0 celen))))
         (grtext -2 (itoa (setq num (1+ num)))))
  (setvar "blipmode" blip)
 malist)
 ; Ŀ
 ;   Doto end                                                              
 ; 

 ; Ŀ
 ;   Setup - make a square grid with cells.                                
 ; 
 (DEFUN C:SETUP ()
  (setvar "cmdecho" 0)
  (setq pa (getpoint "\nUpper Left: "))
  (setq celen (getdist pa "\nCell length: "))
  (setq gridsz (getint "Grid size <10>: "))
  (if (null gridsz) (setq gridsz 10))
  (setq malist (dotto pa celen gridsz))
  (setq pa (polar (polar pa pi (/ celen 2)) (/ pi 2) (/ celen 2)))
  (bomax pa gridsz celen)
  (prompt "\nErase/toggle unneeded cells and run Life.\n")
 (princ))

 ; Ŀ
 ;   Life.                                                                 
 ; 
 (DEFUN C:LIFE (/ cycle num cyc celsub celnam livnum celxst enam procel)
  (command ".redraw")
  (setq cycle 0)
  (if (null malist)
      (progn
           (write-line "No data list - running Setup.\n")
           (c:setup))
      (repeat 120
              (setq num 0)
              (setq cyc (strcat (itoa (setq cycle (1+ cycle))) ":"))
              (while (setq celsub (nth num malist))
                     (setq celnam (cadr celsub))
                     (grtext -2 (strcat cyc (itoa (setq num (1+ num)))))
                     (setq livnum (neigh (car celsub) gridsz))
                     (setq celxst (if (entget celnam) T ()))
                     (if (or (and celxst (or (<= livnum 1) (>= livnum 4)))
                             (and (null celxst) (= livnum 3)))
                         (setq procel (append procel (list celnam)))))
              (while (setq enam (car procel))
                     (entdel enam)
                     (setq procel (cdr procel)))
                     (command ".redraw")))
 (princ))

(prompt "C:LIFE/C:SETUP/C:TOG/C:LREAD/C:LSAVE")
(princ)
